      subroutine alkcor
c
c ********************************************************************
c
c  -- The purpose of this routine is to calculate the total dissolved
c     concentration of component 140, CO3-2, from the user-supplied
c     value of alkalinity.  The alkalinity is stored in t(j) where
c     j = the component index corresponding to id # 140.  Alkalinity
c     as used here means that the value supplied represents the
c     acid-neutralizing capacity of the solution as determined by
c     titrating the solution to the CO2 equivalence endpoint.
c     This corresponds to an operational definition implemented here
c     which is:  The alkalinity is the negative of the TOTH expression
c     when the components are the principal components at the CO2
c     equivalence point.  BEYOND THIS, THE EXACT DEFINITION OF
c     ALKALINITY AS USED IN ANY PARTICULAR PROBLEM DEPENDS ON THE
c     CONTENT OF THE FILE ALK.DBS.
c
c     In general, the principal components of an aqueous solution are
c     those components that result when the following component
c     selection rules are obeyed:
c
c       Choose components for the system in the following order:
c
c        1 - Choose H2O.
c        2 - Choose H+.
c        3 - Choose species with fixed activities, i.e., solids
c            or gases at fixed partial pressures.
c        4 - Choose the most abundant soluble species.
c
c
c     MINTEQA2 has a pre-defined set of components not all of which
c     are the principal components at the CO2 equivalence point, but
c     this does not invalidate the applicability of the operational
c     definition of alkalinity.  It simply means that the alkalinity
c     expression will incorporate species that may also be
c     components.  Practically, the procedure for determining the
c     alkalinity factors needed for carbonate containing species in
c     THERMO.DBS and for entries in ALK.DBS is to write (outside of
c     MINTEQA2) the TOTH expression derived using the principal
c     components at the CO2 equivalence point.  The negative of the
c     coefficient for each of species as it occurs in the TOTH
c     equation is the alkalinity factor for that species.  Stated
c     differently, the contribution of each species in solution to the
c     alkalinity is given by the negative of the stoichiometry of
c     H+ in that species times that species concentration.
c
c     The only species that are implicitly included as contributors to
c     the entered alkalinity are the TYPE 1 component species CO3-2
c     and those TYPE 2 species for which there is a non-zero entry for
c     the alkalinity factor in THERMO.DBS (such as for HCO3-).  For
c     these species, it is not necessary for the user to know the
c     alkalinity factor; the appropriate factor is already in
c     THERMO.DBS.
c
c     If other species are to be included as contributors to
c     alkalinity, the id number and the proper alkalinity factor, as
c     given by the TOTH expression derived using the principal
c     components at the CO2 equivalence point, must be entered in
c     ALK.DBS.  Entries for species containing CO3-2 MUST NOT be
c     included in that file.  Suppose for example that the sample you
c     wish to model and for which you've measured the alkalinity
c     contains appreciable dissolved phosphate.  You may wish to
c     account for the phosphate contribution to the measured
c     alkalinity so as to arrive at a more correct value for total
c     dissolved carbonate.  Procedure:
c
c     1 -   Principal components at CO2 equivalence point:
c           (H+, CO2, H2PO4-, ..., other components)
c
c     2 -   TOTH = (H+) - (HCO3-) - 2(CO3-2) + (H3PO4) - (HPO4-2)
c                  - 2(PO4-3) + ... other species
c
c     3 -   Noting that the pK1 for H3PO4 is 2.2, we might choose to
c           omit that species altogether due to its negligible
c           concentration above pH 4.  Retaining it here for
c           illustration, entries in ALK.DBS would look like this:
c
c                      3305802 -1.00
c                      3305800  1.00
c                          580  2.00
c                      3300020  1.00
c                          330 -1.00
c
c           where the 7 and 3 digit numbers are species id numbers:
c                      3305802 = H3PO4
c                      3305800 = HPO4-2
c                          580 = PO4-3
c                      3300020 = OH-
c                          330 = H+
c           and the rightmost digit in each id number is in column 7
c           and the first line in the file holds the first entry.
c
c
c     Thus, in this case, the alkalinity equation to which the value
c     input for alkalinity corresponds is:
c
c     alk = [HCO3-] + 2[CO3-2] - [H3PO4] + [HPO4-2] + 2[PO4-3]
c            + [OH-] - [H+]
c
c     In most natural systems, the phosphate species are at much
c     lesser concentration than the carbonate and can thus be
c     neglected.  But, as shown above, the user controls the
c     alkalinity equation by preparing ALK.DBS.
c
c     If this same file is used in a MINTEQA2 run for which component
c     id # 580 (PO4-3) is not included, then those entries in ALK.DBS
c     that involve component 580 are simply ignored and the alkalinity
c     equation that would be used is:
c
c     alk = [HCO3-] + 2[CO3-2] + [OH-] - [H+]
c
c     Let the contribution of non-carbonate containing species to the
c     input alkalinity be called non-carbonate alkalinity.  Then, the
c     dissolved total inorganic carbon (TIC) is given by
c
c                    TIC = alk - excrb - noncrb + c(iyh2co3)
c
c     where  alk = The input alkalinity value converted to eq/L.
c
c          excrb = The total number of excess equivalents of acid
c                  consumed per mole of carbonate containing species,
c                  (total meaning summed over all such species).
c                  For each carbonate species, the alkalinity factor
c                  gives the total number of equivalents of acid
c                  consumed per mole.  Therefore, the excess
c                  equivalents for each such species is given by the
c                  difference between the alkalinity factor and the
c                  stoichiometry of CO3-2 in that species times the
c                  the number of moles, that is, times the
c                  concentration (per liter basis).
c
c         noncrb = The total number of equivalents of non-carbonate
c                  alkalinity (total meaning summed over all species
c                  contributing to non-carbonate alkalinity).
c                  For each non-carbonate species, the alkalinity
c                  factor is the number of equivalents of acid
c                  consumed per mole.  Therefore, the number of
c                  equivalents of non-carbonate alkalinity for
c                  each such species is the alkalinity factor times
c                  the species concentration (per liter basis).
c
c     c(iyh2co3) = The concentration of H2CO3(aq).
c
c
c  -- REFERENCES:
c         Morel, Francois M.M., PRINCIPLES of AQUATIC CHEMISTRY,
c                Wiley, New York, 1983, pp. 105,131-141.
c
c *********************************************************************
c
      include 'MINTEQA2.INC'
c
      real*8 alk,excarb,fact,factor
c
      dimension ialk(30),fact(30),iidt(30)
      include 'CONST.INC'
c
c
c  -- Addresses of relevant species in component space (x) and species
c     space (y).
      ixco3 = iadx(140)
      iyco3 = iady(140)
      iyh2co3 = iady(3301401)
c
c  -- The input value for alkalinity, t(ixco3), is initially
c     in whatever units the user has chosen in PRODEFA2 and is
c     converted to mol/L in Subroutine PREP.  Convert to eq/L by
c     by multiplying by 2.  Also, the alkalinity factor for the
c     component species CO3-2 is 2.0; alkalinity factors for TYPE
c     2 species are in THERMO.DBS.
      if (iter.eq.0) then
         alk = t(ixco3)*2.0d0
         alkfct(iyco3) = 2.0
      endif
c
c  -- Compute the excess equivalents of acid that can be neutralized
c     by HCO3-, CO3-2, and any other aqueous species having non-zero
c     carbonate stoichiometry for which the components are present
c     and the alkalinity factor in the database THERMO.DBS is
c     non-zero.
      excarb = 0.0d0
      ll = nn(1)+nn(2)
      do 100 i = 1, ll
         if (alkfct(i).lt.D0MIN) go to 100
         excarb = excarb + c(i) * (alkfct(i)-a(i,ixco3))
  100 continue
c
c  -- Read the noncarbonate alkalinity file ALK.DBS on the first trip
c     through this routine only.  No need to re-read it on subsequent.
c     iterations.  Assignment of ALK.DBS to lun10 and file OPEN occurs
c     in Program MINTEQA2.
      if (iter.eq.0) then
         i = 0
         rewind lun10
  110    read (lun10,9040) idt,factor
         if (idt.eq.0) go to 120
         k = iady(idt)
         if (k.eq.0) go to 110
         i = i+1
         ialk(i) = k
         iidt(i) = idt
         fact(i) = factor
         go to 110
  120    ii = i
      endif
c
c  -- Compute the total number of equivalents of acid-neutralizing
c     capacity represented by non-carbonate species (i.e., those
c     species which are included as TYPE 1 or TYPE 2 species in this
c     problem and for which entries are present in ALK.DBS).
      noncrb = 0.0d0
      if (ii.eq.0) go to 150
      do 130 i = 1, ii
c
c  -- The bounds on the following IF block seem ridiculously large;
c     If c(ialk(i)) is anywhere near 1e37, we've got big problems.
         if (dabs(c(ialk(i))).lt.1.0d37) then
           noncrb = noncrb + c(ialk(i)) * fact(i)
         end if
  130 continue
c
c  -- The following calculation arrives finally at the main objective:
c     to use the known value of alkalinity, the computed excess
c     acid-neutralizing capacity represented by carbonate species, the
c     computed acid-neutralizing capacity represented by non-carbonate
c     species, and the concentration of H2CO3 to compute the total
c     dissolved inorganic carbon in the system.
c  -- The bounds on the following IF block also seem ridiculously
c     large for the same reason as above.
      if (excarb.lt.1.0d37.and.c(iyh2co3).lt.1.0d37) then
         t(ixco3) = alk - excarb - noncrb + c(iyh2co3)
      else
         t(ixco3) = alk
      endif
c
c
      if (idebug.eq.2) then
         write (lunout,9010)
         do 140 i = 1, ii
            write (lunout,9020) iidt(i),fact(i),idx(ialk(i)),idy(ialk
     *        (i)),c(ialk(i)),x(ialk(i)),gamma(ialk(i)),ialk(i)
  140    continue
         write (lunout,9030) noncrb,excarb
      endif
c
c  -- Prevent total dissolved inorganic carbon from becoming less than
c     or equal to zero during iterations.
  150 if (t(ixco3).le.0.0) t(ixco3) = alk
c
      return
c
 9010 format ('0',5x,'ALK ID ',' FACTOR',2x,'  IDX  ',2x,'  IDY  ',2x
     *   ,'     C    ',2x,'     X    ',2x,'  GAMMA   ',2x,'IALK ')
 9020 format (5x,i7,2x,f5.2,2x,i7,2x,i7,2x,1pe10.3,2x,1pe10.3,2x,
     *   0pf10.5,2x,i3)
 9030 format (5x,'NONCRB= ',1pe11.4,' CO3ALK = ',e11.4)
 9040 format (i7,1x,f5.2)
c
      end
