      subroutine composit (iy,vlig)
c     *************************************************************
c     J.D. Allison  October 1991
c
c     This subroutine computes the concentration of a metal/ligand
c     complex where the ligand is a composite of sites representing
c     various functional groups such that the material as a whole
c     exhibits a continuous distribution of log K's for binding
c     a given metal.  The distribution is assumed to be Gaussian
c     with a known mean log K and standard deviation (sigma).
c     Integration over the Gaussian distribution is by means of the
c     Gaussian-Hermite quadrature using a pre-defined set of
c     points and weights.
c
c     The contributions to the gradients are also calculated.
c     These are added to the appropriate elements of the z(i,j)
c     array and are referred to here as "partial gradients".
c     They are computed by taking partial derivatives of the
c     concentration of the metal/ligand complex with respect to
c     the activity of each component complexing the ligand.
c
c
c     --------------------------------------------------------------
c     INPUT**INPUT**INPUT**INPUT**INPUT**INPUT**INPUT**INPUT**INPUT
c
c       iy = Corresponds to the index i in global arrays c(i),
c            a(i,j), idy(i), etc.  The complex whose concentration
c            we seek is the species whose id number is idy(iy).
c
c       Parameters and common block variables in MINTEQA2.INC used
c       as indirect input variables:
c
c            idclf   =  The lower boundary of the range of 3-digit
c                       component id numbers for composite ligands.
c
c            idcll   =  The upper boundary of the range of 3-digit
c                       component id numbers for composite ligands.
c                       Because the 7-digit species id number for
c                       ALL complexes involving a composite
c                       ligand should have been entered in the
c                       database with the component id of the ligand
c                       as its leftmost digits, ALL such species
c                       id numbers MUST fall within the range
c                       idclf*10000 thru idcll*10000 + 9999.
c
c            idrxcl  =  An array containing the 7-digit id numbers
c                       of all complexes read from the database
c                       involving the composite ligand.  Values of
c                       array elements in idrxcl are assigned in
c                       subroutine INPUT.
c
c            nrxcl   =  The number of entries in idrxcl.  The total
c                       number of reactions in which the composite
c                       ligand is a reactant.  The value of nrxcl is
c                       assigned in subroutine INPUT.
c
c            idxcl   =  An array containing the component id numbers
c                       of all components involved in composite
c                       ligand reactions.  Does not include the
c                       id number of the composite ligand component
c                       itself.
c
c           nidxcl   =  The number of entries in idxcl.  The number of
c                       components involved in reactions with the
c                       composite ligand (not counting the ligand
c                       itself).
c
c            stoica  =  A 2-d array having one row for each complex
c                       involving the composite ligand (thus, each
c                       row corresponds to an entry in idrxcl and there
c                       are nrxcl rows).  Each column represents
c                       a component involved in a reaction with
c                       the composite ligand (the ligand itself
c                       has no corresponding column).  Thus, each
c                       column is identified with an entry in the
c                       array idxcl and there are nidxcl columns.
c                       The entries themselves give the stoichiometry
c                       of the component (represented by that column
c                       index) in the species (represented by the
c                       row index).
c
c            jmaxcl  =  An integer array containing the number of non-
c                       zero entries along a particular row of stoica.
c
c            jptacl  =  A 2-d integer array containing the column indices
c                       of the non-zero elements of a particular row
c                       of stoica.
c
c            gamma   =  A global array containing the common log of
c                       the activity coefficients as computed in
c                       Subroutine ACTVTY.
c
c            idy     =  A global array containing the id numbers
c                       of ALL species; includes components as
c                       free species.
c
c            idx     =  A global array containing the 3-digit id
c                       numbers of ALL components involved in
c                       this problem.
c
c            nnn     =  The total number of components in this
c                       problem.
c
c            sigma   =  A variable containing the standard deviation
c                       in log K for a complex involving a
c                       composite ligand.  In this implementation
c                       of the composite ligand model, all reactions
c                       have identically the same value of sigma.
c
c            t       =  A global array containing the total
c                       dissolved concentrations of ALL components
c                       involved in this problem.
c
c
c     --------------------------------------------------------------
c     LOCAL**LOCAL**LOCAL**LOCAL**LOCAL**LOCAL**LOCAL**LOCAL**LOCAL
c
c       npts   = The total number of points and weights (32).
c
c       zp     = An array containing the Gaussian-Hermite points.
c
c       wt     = An array containing the Gaussian-Hermite weights.
c
c       sqr2   = Square root of 2.
c
c       wtsum  = A normalization factor for the integration equal to
c                the sum of the weights and to the reciprocal of pi.
c
c       jxlig  = The array index address of the composite ligand
c                component.
c
c       xnum   = An array containing the numerator portion of the
c                expression for the concentration of metal/ligand
c                complex.  A value of xnum is computed for each
c                point zp along the integration path by multiplying
c                by the appropriate weight.
c
c       xdenom = An array containing the denominator portion of the
c                expression for the concentration of metal/ligand
c                complex.  A value of xdenom is computed for
c                each point zp along the integration path.
c
c       cxnum  = The resulting numerator in the expression for bound
c                metal if the individual quoients corresponding to
c                each point in the Gaussian quadrature integration
c                are summed after finding a common denominator.
c
c       cxdenom= The common denominator of all the individual quoients
c                of the Gaussian quadrature.
c
c       gkclc  = An array to which is assigned the corrected mean
c                log K values pertaining to each reaction involving
c                the composite ligand.  "Corrected" means multiplied by
c                the activity coefficient pertaining to the free ligand
c                and divided by the activity coefficient pertaining
c                to the metal-ligand complex.  The uncorrected
c                mean log K values are retained in the global array
c                gkcl.
c
c       term   = An array whose elements contain the terms of the
c                denominator that arise from all the composite
c                ligand complexes including the one for which we are
c                calculating the concentration in a call to this
c                routine.  Each term is represented by the left
c                hand side of a mass action equation for the
c                corresponding reaction given that the equation is
c                arranged so that only one product, the metal/ligand
c                complex, appears on the right-hand side.
c
c       conc   = The intermediate result of the Gaussian-Hermite
c                quadrature integration and finally, the concentration
c                of the complex whose id is idy(iy); i.e., the answer
c                we seek.  The common log of conc is returned to the
c                calling program unit as the answer.
c
c       tmp1   = A temporary storage buffer used for intermediate
c                results in calculating the terms contributing to
c                xdenom and in calculating the partial derivatives.
c
c       tmp2   = A temporary storage buffer used for intermediate
c                results in calculating the partial derivatives.
c
c
c
c
c     --------------------------------------------------------------
c     OUTPUT**OUTPUT**OUTPUT**OUTPUT**OUTPUT**OUTPUT**OUTPUT**OUTPUT
c
c     Primary output:
c
c       vlig = The log concentration of the metal/ligand complex
c              whose id number is idy(iy).
c
c
c     Variables in MINTEQA2.INC common blocks modified by COMPOSIT:
c
c       z(j,k) =  The gradients or partial derivatives.  More
c                 specifically, the change in the mass imbalance of
c                 component j as a function of change in the
c                 activity of component k.  Only those gradients
c                 involving components that complex the composite
c                 ligand are modified.  Note that for every
c                 component,  each z(j,k) is actually a summation
c                 of contributing terms.  Each term is computed by
c                 taking the partial derivative of a species
c                 concentration (the species must have non-zero
c                 stoichiometry in component j, else the computation
c                 is pointless).  The derivative is taken with respect
c                 to the activity of component k.  The "partial
c                 gradients" computed here are for those
c                 terms where the species of interest is the
c                 metal/ligand complex with id number idy(iy).
c
c
c
c
c     ***************************************************************
c     BEGIN**BEGIN**BEGIN**BEGIN**BEGIN**BEGIN**BEGIN**BEGIN**BEGIN
c     ***************************************************************
c
      real*8 zp(32), wt(32), gkclc(14), term(14), xdenom(32), xnum(32),
     &       conc, vlig, wtsum, tmp1, tmp2, cxdenom, cxnum
c
      include 'MINTEQA2.INC'
c
c
c  -- The set of 32 points for the Gaussian-Hermite quadrature:
      data (zp(i), i = 1, 32) / 0.7125813909830728E+01,
     *        0.6409498149269661E+01,
     *        0.5812225949515914E+01,
     *        0.5275550986515880E+01,
     *        0.4777164503502596E+01,
     *        0.4305547953351199E+01,
     *        0.3853755485471445E+01,
     *        0.3417167492818571E+01,
     *        0.2992490825002374E+01,
     *        0.2577249537732317E+01,
     *        0.2169499183606112E+01,
     *        0.1767654109463202E+01,
     *        0.1370376410952872E+01,
     *        0.9765004635896828E+00,
     *        0.5849787654359324E+00,
     *        0.1948407415693993E+00,
     *       -0.1948407415693993E+00,
     *       -0.5849787654359324E+00,
     *       -0.9765004635896828E+00,
     *       -0.1370376410952872E+01,
     *       -0.1767654109463202E+01,
     *       -0.2169499183606112E+01,
     *       -0.2577249537732317E+01,
     *       -0.2992490825002374E+01,
     *       -0.3417167492818571E+01,
     *       -0.3853755485471445E+01,
     *       -0.4305547953351199E+01,
     *       -0.4777164503502596E+01,
     *       -0.5275550986515880E+01,
     *       -0.5812225949515914E+01,
     *       -0.6409498149269661E+01,
     *       -0.7125813909830728E+01/
c
c
c  -- The set of 32 points for the Gaussian-Hermite quadrature:
      data (wt(i), i = 1, 32) / 0.7310676427384062E-22,
     *        0.9231736536518204E-18,
     *        0.1197344017092854E-14,
     *        0.4215010211326414E-12,
     *        0.5933291463396681E-10,
     *        0.4098832164770878E-08,
     *        0.1574167792545590E-06,
     *        0.3650585129562368E-05,
     *        0.5416584061819989E-04,
     *        0.5362683655279717E-03,
     *        0.3654890326654427E-02,
     *        0.1755342883157342E-01,
     *        0.6045813095591263E-01,
     *        0.1512697340766425E+00,
     *        0.2774581423025300E+00,
     *        0.3752383525928024E+00,
     *        0.3752383525928024E+00,
     *        0.2774581423025300E+00,
     *        0.1512697340766425E+00,
     *        0.6045813095591263E-01,
     *        0.1755342883157342E-01,
     *        0.3654890326654427E-02,
     *        0.5362683655279717E-03,
     *        0.5416584061819989E-04,
     *        0.3650585129562368E-05,
     *        0.1574167792545590E-06,
     *        0.4098832164770878E-08,
     *        0.5933291463396681E-10,
     *        0.4215010211326414E-12,
     *        0.1197344017092854E-14,
     *        0.9231736536518204E-18,
     *        0.7310676427384062E-22/
c
c
c
      npts = 32
      wtsum = dsqrt (4.0d0 * datan (1.0d0))
      sqr2 = dsqrt(2.0d0)
c
c  -- Find the array index address of the composite ligand component.
      do 10 j = 1, nnn
        if (b(iy,j).gt.0.0d0
     &     .and.(idx(j).ge.idclf.and.idx(j).le.idcll)) jxlig = j
  10  continue
c
c  -- Find the array index address (irxn) of the complex
c     whose concentration is to be computed.
      do 20 i = 1, nrxcl
        if (idy(iy) .eq. idrxcl(i)) irxn = i
  20  continue
c
c  -- Find the array index address in species space of all species
c     competing for the composite ligand.  We need to correct their
c     log K values for ionic strength by using previously computed
c     (Subroutine ACTVTY) log gamma's.   Store the corrected log K 's
c     in local array gkclc, preserving the originals in gkcl for use
c     in future corrections.
      do 30 k = 1, nrxcl
        n1 = iady(idrxcl(k))
        n2 = iady(idx(jxlig))
        gkclc(k) = gkcl(k) - gamma(n1) + gamma(n2)
  30  continue
c
c
c     Each element of the array term(k) corresponds to a term
c     in the denominator xdenom.  The denominator is a summation
c     of these terms including a term involving the complexing component
c     of the species for which we seek the concentration (that term is
c     also the numerator), as well as all other competing components.
c
      do 40 k = 1, nrxcl
        tmp1 = gkclc(k)
        do 50 j = 1, jmaxcl(k)
          jjj = jptacl(k,j)
          n = iady(idxcl(jjj))
          tmp1 = tmp1 + stoica(k,jjj)*(gc(n)+gamma(n))
  50    continue
        term(k) = 10.0d0**tmp1
  40  continue
c
c
c  -- Main computation loop
c
c     This loop computes and accumulates conc = xnum / xdenom for each
c     index i.  Values of xnum and xdenom are computed to
c     correspond with each value of zp(i) and wt(i).  The quoient
c     xnum/xdenom is summed (accumulated) in the variable conc, the
c     concentration of the species of interest.
c
      conc = 0.0d0
c
      do 60 i = 1, npts
        xnum(i) = term(irxn) * t(jxlig) / wtsum
c
        xdenom(i) = 1.0d0 / 10.0d0**(sigma * sqr2 * zp(i))
c
        do 70 k = 1, nrxcl
          xdenom(i) = xdenom(i) + term(k)
  70    continue
c
c
c  -- Compute the value of the quoient
        xnum(i) = xnum(i) * wt(i)
        conc = conc + xnum(i) / xdenom(i)
c
  60  continue
c
c  -- COMPUTE DERIVATIVES
c
c  -- Compute the common denominator of the overall quoient.  This
c     is just the product over all point denominators xdenom(i).
      cxdenom = xdenom(1)
      do 110 i = 2, npts
        cxdenom = cxdenom * xdenom(i)
 110  continue
c
c
c  -- Compute the numerator of the overall quoient.  This is
c     computed by:
c
c          npts                  npts
c          SUM  [xnum(i)   *     PROD (xdenom(j)) ]
c          i=1                j=1,j.ne.i
c
c     This can be computed by accumulating the sum over i of each
c     xnum(i) times cxdenom/xdenom(i).
      cxnum = 0.0d0
      do 120 i = 1, npts
        cxnum = cxnum + xnum(i) * cxdenom/xdenom(i)
 120  continue
c
c
c  -- Compute the partial derivatives using the quoient rule:
c
c
c
c            d (conc)       d  (cxnum/cxdenom)
c           ---        =   ---
c           dx             dx
c
c
c
c                            d (cxnum)        cxnum  d (cxdenom)
c                           ---                     ---
c                      =    dx           -          dx
c                         ------------       -------------------
c                           cxdenom              cxdenom**2
c
c
c     The second term of this expression is divided by xdenom
c     squared, a very large number.  Thus, this term is negligible
c     and will be omitted from the calculations.
c
c  -- For those components for which the metal/ligand concentration
c     just computed enters into the mass balance equation:
      do 80 j = 1, jmaxcl(irxn)
        jjj = jptacl(irxn,j)
        jj = iadx(idxcl(jjj))
c
c  -- The partial derivative of the metal/ligand concentration is
c     taken with respect to the activity of each component having
c     non-zero stoichiometry in species irxn (i.e., that complexes
c     the ligand):
        do 90 k = 1, jmaxcl(irxn)
          kkk = jptacl(irxn,k)
          kk = iadx(idxcl(kkk))
          n = iady(idxcl(kkk))
c  -- Take the derivative of cxnum with respect to component idxcl(kkk)
c     and then divide by cxdenom.  This is the first term in the
c     quoient rule expression.  The second term is negligible and
c     is omitted from the calculations.
          tmp1 = (stoica(irxn,kkk) * cxnum / (c(n)*10.0d0**gamma(n)))
     *            / cxdenom
c
c     ****************************************************************
c     SPECIAL NOTE:
c
c     The following code segment shows how to compute the second
c     term should it ever prove necessary to do so.  Test runs with
c     this code not commented out show that tmp2 will be very
c     small.  (A value of 1e-80 was the largest tmp2 observed in
c     the tests).
c
c  -- Take the derivative of cxdenom, with respect to component
c     idxcl(kkk), multiply by cxnum, and divide by cxdenom**2.  This
c     is the second term in the quoient rule expression.
c          tmp2 = 0.0d0
c          do 100 m = 1, nrxcl
c            tmp2 = tmp2 + stoica(m,kkk)*term(m)/(c(n)
c     *             *10.0d0**gamma(n))
c 100      continue
c          tmp2 = cxnum * tmp2 / cxdenom**2.0d0
c      ***************************************************************
c
c
c  -- Add the partial gradient to the current gradient after
c     multiplying by the stoichiometry of this component in the
c     mass balance expression.  The partial gradient is the
c     difference tmp1-tmp2, but since tmp2 is negligible, we
c     just use tmp1.
          z(jj,kk) = z(jj,kk) + tmp1 * stoica(irxn,jjj)
  90    continue
  80  continue
c
c
c
c
c  -- Take the common log for return to calling program unit.
      vlig = dlog10 (conc)
c
      return
c
      end
