{ -------------------------------------------------------------------------- }
{ -- STX3_ANA.PAS contains a pseudo analytic solution to the model.   This-- }
{ -- is taken from the file A3ATMDEP.PAS, developed by Joe Helfand.   This - }
{ -- model is called only by stx3_run -------------------------------------- }
{ -------------------------------------------------------------------------- }


{ ----------------------- Created 7/10/90 by DC ---------------------------- }
{ ------------------------- Modification lists ----------------------------- }

   { 7/26/90  Speeding it Up - DC }

{ -------------------------------------------------------------------------- }
{ -------------------------------------------------------------------------- }
unit Stx3_ana;

interface

uses crt,textutil;

type
 real = system.double;   { forces 8087 }

 var para,parb,parc : real;

function  find_analytical( Kp1,     {water column decay rate, aggregate  day-1}
                           kd1,
                           kp2,     {sediment decay rate, aggregate day-1}
                           kd2,
                           KL,     {diffusion velocity m/day}
                           U1,     {velocity m/day}
                           CT10,   {water column total toxicant xx/L, at x=0}
                           Wrs,    {resuspension velocity m/day}
                           Ws,     {settling velocity m/day, must be >0}
                           m10,    {water column solids mg/L, at x=0}
                           m2,     {sediment solids mg/L}
                           pi1,    {water column partition coefficient L/mg}
                           pi2,    {sediment partition coefficient L/mg}
                           H1,     {water column depth m}
                           H2,     {active sediment depth m}
                           AD,     {atmospheric deposition xx/m/day}
                           q,      {total reach flow L/day}
                           xf : real; first_time : boolean) : real;   {Distance from beginning of reach }
         { Finds the analytical solution for total water toxicant }

implementation

{PseudoAnalytic solution of the SMPTOX3 model for a single reach, including
atmospheric deposition}
{initial values must include waste stream with assumed instantaneous mixing}
{for degenerate cases Ws=0 or m2=0 will get division by zero}
{Joseph Helfand  July, 1990}
{ST3-1}


function  find_analytical( Kp1,                  {water column decay rate, aggregate  day-1}
                           kd1,
                           kp2,                  {sediment decay rate, aggregate day-1}
                           kd2,
                           KL,                  {diffusion velocity m/day}
                           U1,                  {velocity m/day}
                           CT10,                {water column total toxicant xx/L, at x=0}
                           Wrs,                 {resuspension velocity m/day}
                           Ws,                  {settling velocity m/day, must be >0}
                           m10,                 {water column solids mg/L, at x=0}
                           m2,                  {sediment solids mg/L}
                           pi1,                 {water column partition coefficient L/mg}
                           pi2,                 {sediment partition coefficient L/mg}
                           H1,                  {water column depth m}
                           H2,                  {active sediment depth m}
                           AD,                  {atmospheric deposition xx/m/day}
                           q,                   {total reach flow L/day}
                           xf : real;first_time : boolean) : real;   {Distance from beginning of reach }

VAR
   xf2,fp2,fd2,CT1,CT1xf,CT1xf2,xx,k2:real;
   outfile : text;
   savetime_1 : real;
   savetime_2 : real;
   savetime_3 : real;
   savetime_4 : real;
   savetime_5 : real;
   savetime_6 : real;
   savetime_7 : real;
   savetime_8 : real;
   savetime_9 : real;
   i:integer;

FUNCTION BS(q,fd2,fp2:real):real;
{this function evaluates part of an integral at boundary limits q}
VAR
   xx:real;
BEGIN  {BS}
      xx:= KL*fd2+K2*H2 + savetime_9 * (Savetime_4 + pi1*fp2*Wrs);
      xx:= xx + Savetime_3 * (Savetime_4 + savetime_6) * q;
      BS:= xx + (savetime_5)*(Savetime_3)*(Savetime_3)*q*q;
END;  {BS}

PROCEDURE Calc_CT1(x,fd2,fp2,para,parb,parc:real; VAR CT1:real);

VAR
   A,B1,B2,C,D1,D2,E,F:real;
   cc,dd,ee,ff,gg,hh : real;
   q,alpha:real;

BEGIN      {Calc_CT1}
{solution is ln(CT1/CT10) = A + B1 + B2 + C + D1 + D2 + para*x +
                            parb*x*x/2.0 + parc*x*x*x/3.0}
{then CT1 = CT10*exp(A + B1 + B2 + C + D1 + D2 + para*x +
                     parb*x*x/2.0 + parc*x*x*x/3.0)}
{E, F, q, and alpha are intermediate quantities}
{groups are dimensionless except F, which is day/m,
and alpha, which is velocity squared}

{calculate q}
q:= exp(-Ws*x/(savetime_8));

{calculate E}
E:= ln(((Savetime_1)+pi1*(SaveTime_2)*q) /
      ((Savetime_1)+pi1*(SaveTime_2)));

{calculate A}
{ A:=  -K1*x/U1; }
 A := 0;  { because k1 not used }

{calculate B1 and B2}

dd := (Ws+H1*Kp1)/Ws;
B1:= -((Wrs*m2*pi1)/(Savetime_1)) * (E - ln(q)) * dd;
B2:= E*dd;  { for separate rate Kp1 }


{calculate C}

C:= -(x + (savetime_8/Ws)*E) * (kl+h1*kd1)  / (savetime_8*(Savetime_1)/Ws) ;

{calculate alpha}
alpha:= 4.0*((KL*fd2+K2*H2) + (Savetime_4)*(savetime_9) +
        pi1*fp2*Wrs*savetime_9)*(savetime_5)*SQR(Savetime_3) -
        SQR((Savetime_4)*(Savetime_3) +
        (savetime_6)*(Savetime_3));

{calculate F}
{solution method depends on alpha}
IF alpha=0.0 THEN
begin
     writeln('  alpha = 0, solution invalid');
     halt;
end;
IF alpha>0.0 THEN
   BEGIN  {if}
          cc := SQRT(alpha);

          ee := Savetime_3 * ( Savetime_4 + savetime_6);
          ff := 2.0 * savetime_5 * SQR(Savetime_3);
          dd := (ff + ee) / cc;
          gg := (ff*q + ee) / cc;

          F := (2.0 / cc) * (Arctan(gg) - Arctan(dd));
   END  {if}
    ELSE
   BEGIN  {else}
          cc := SQRT(-alpha);
          ee := Savetime_3 * (Savetime_4 + savetime_6);
          ff := 2.0 * savetime_5 * SQR(Savetime_3);
          dd := (ff + ee - cc) / (ff + ee + cc);
          hh := (ff * q + ee);
          gg := (hh - cc) / (hh + cc);
          F  := (1.0 /cc) * ( ln(gg) - ln(dd) );
   END;   {else}

{calculate D1 and D2}
D2:= -pi1/Ws*(savetime_7)*(SaveTime_2)*F;
IF (Wrs=0) AND (KL=0) THEN
   D1:=0
   ELSE
   BEGIN   {else}
   D1:= -(savetime_7)*(Wrs*m2*pi1 + KL)/Ws;
   D1:= D1/(2.0*((KL*fd2+K2*H2)+(Savetime_4)*(savetime_9) +
           (pi1*fp2*Wrs*savetime_9)));
   D1:= D1*(ln(q*q*BS(1.0,fd2,fp2)/BS(q,fd2,fp2)) -
        F*((Savetime_4)*(Savetime_3) +
        (savetime_6)*(Savetime_3)));
   END;   {else}

{calculate total water column concentration at x}
CT1:= CT10*exp(A + B1 + B2 + C + D1 + D2 + para*x + parb*x*x/2.0 +
      parc*x*x*x/3.0);

END; {calc_CT1}

FUNCTION KK(x,fp2,fd2:real):real;
VAR
   m1,fp1,fd1,temp,temp2,dd:real;
BEGIN
     dd := EXP(-(Ws * x)/savetime_8);
     m1:= m10 * dd + (savetime_9) * (1.0-dd);
     fd1:= 1.0/(1.0 + m1*pi1);
     fp1:= 1.0 - fd1;
     temp2:= -(kd1*fd1 + kp1*fp1)/U1 - Ws*fp1/(savetime_8) - KL*fd1/(savetime_8);
     temp := ((Ws*fp1+KL*fd1)/(Ws*m1*fp2/m2 + KL*fd2 + K2*H2));
     KK:= temp2 + (((savetime_7)/(savetime_8)) * temp);
END; {KK}

var dd : real;
BEGIN  {a3}
{x can be any point in the reach}
xf2:= xf / 2.0;

{calculate sediment dissolved and particulate fractions}
fd2:= 1.0/(1.0 + m2*pi2);
fp2:= 1.0 - fd2;
k2 := (kp2*fp2) + (kd2*fd2);

savetime_1 := ws + wrs * m2 * pi1;
savetime_2 := m10 * ws - Wrs * m2;
savetime_3 := m10 - wrs * m2 / Ws;
savetime_4 := ws * fp2/m2 + pi1*kl*fd2 + pi1 * k2 * h2;
savetime_5 := Ws*pi1*fp2/m2;
savetime_6 := pi1*fp2*2.0*Wrs;
savetime_7 := Wrs*fp2 + KL*fd2;
savetime_8 := H1*U1;
savetime_9 := Wrs*m2/Ws;

{ xx:= (AD*xf)/(q*CT10); }
if ct10 = 0 then
   xx := 0
else
    xx := abs((ad/q)/(KK(0,fp2,fd2)*ct10));

if first_time then
begin
     para:=0;
     parb:=0;
     parc:=0;

     IF xx < 0.25 THEN
     BEGIN
          {start convergence loop}

          IF (AD<>0) and (ct10 <> 0) THEN
          FOR i:=1 TO 5 DO
          BEGIN  {for}
               Calc_CT1(xf2,fd2,fp2,para,parb,parc,CT1xf2);
               Calc_CT1(xf,fd2,fp2,para,parb,parc,CT1xf);
               para:= AD / (q*CT10);
               parc:= AD * (2.0/CT10 + 2.0/CT1xf - 4.0/CT1xf2) / (q*xf*xf);
               parb:= AD / (q*xf) * (1.0/CT1xf - 1.0/CT10) - parc*xf;
          END;  {for}

          {final value}
     END
     ELSE
     BEGIN
          {start convergence loop}
          FOR i:=1 TO 10 DO
          BEGIN
               dd := xf * xf;
               CT1xf:= CT10 + (para+AD/q)*xf + parb*dd/2.0 + parc*dd*xf/3.0;

               dd := xf2 * xf2;
               CT1xf2:= CT10 + (para+AD/q)*xf2 + parb*dd/2.0 + parc*dd*xf2/3.0;

               para:= KK(0,fp2,fd2)*CT10;
               dd := (KK(xf,fp2,fd2)*CT1xf-para);
               parc:= (KK(xf2,fp2,fd2)*CT1xf2 - para - xf2/xf * dd) /(xf2*xf2-xf2*xf);
               parb:= dd / xf - parc*xf;
          END;
     end;   { End Switch }
end; { end first_time }

{final value}
dd := xf * xf;
IF xx < 0.25 THEN
   Calc_CT1(xf,fd2,fp2,para,parb,parc,CT1)
else
   CT1:= CT10 + (para+AD/q)*xf + parb*dd/2.0 + parc*dd*xf/3.0;

find_analytical := ct1;


END;  {a3}

end.  { end implementation }

