{ -------------------------------------------------------------------------- }
{ -- STX3_RUN contains the heart of the the SMPTOX3 Model.  This unit is --- }
{ -- called from unit STX3.PAS and calls procedures in textutil for -------- }
{ -- output of results. ---------------------------------------------------- }
{ -------------------------------------------------------------------------- }


{ ----------------------- Created 5/31/90 by DC ---------------------------- }
{ ------------------------- Modification lists ----------------------------- }

{ 6/1/90   Finishing Run_Model procedure - DC }
{ 6/7/90   Adding some comments -DC }
{ 6/8/90   Saving suspended solids, added mix of concentrations -DC }
{ 6/12/90  Adding Sensitivity Analysis and calibration - DC }
{ 7/3/90   Adding call to modify discharge in wla anlaysis - DC }
{ 7/10/90  Added "I am thinking Leave Me alone" display - DC }
{ 7/11/90  Adding Analytical Solution, Forcing to store ~310 results - DC }
{ 7/18/90  Changing description of simulation waiting screen - DC }
{ 7/24/90  Changed Calculation of run_m1, at suggestion of JH, to reduce }
{             round-off error - DC }
{ 7/25/90  Adding multiple simulations option. - DC }
{ 8/13/90  Redoing Memory management scheme - DC }
{ 10/8/90  Found a bug in equations, reversed partition coefficients - DC }
{ 10/9/90  Moving to analytical solution for non-deposition cases - DC }
{ 1/5/91   Adding SMPTOX123 capabilities - DC }
{ 4/5/91   Fixed line 989, sens_u[1] should be sens_u[i] - DC }
{ 10/10/91 Changing version to SMPTOX3 2.0 - DC }
{ 10/10/91 Changing escaping routines out of wla and sens analysis - DC }
{ 10/17/91 Changing escaping for many_model - DC }
{ 01/23/91 Added Dispose() to many_model exit for insufficient memory - TS }

{ -------------------------------------------------------------------------- }
{ -------------------------------------------------------------------------- }
unit stx3_run;

interface

uses
     stx3_grp,
     textutil,
     scrnutil,
     stx3_var,
     stx3_msc,
     stx3_fil,
     stx3_scr,
     stx3_ana,
     dos,
     crt;

procedure Run_Model;   {  runs the model for the given reach }
procedure Many_Model(file_chosen,been_before:boolean);    { runs the model for multiple input files }

implementation
const number_reports = max_graph_size + 10;
const mday_to_mmyear = 365000.0;

var
    this_piece : real;
    per_piece  : integer;
    next_piece : integer;
    next_dx    : real;
    report_inc : real;
    initial_m1 : reach_var;
    initial_ct : reach_var;
    time_a : real;
    time_n : real;
    loop_here  : integer;
    hour,min,sec,s100 : word;

  { ------- disposes of dynamically allocated data -------- }
  procedure clean_up;
  var i,j : integer;
  begin
        if graph_type = many_plot then
           for i := 1 to nplots do
                dispose(graph_buf[i])
        else
        if graph_type = Sensitivity_plot then
           for i := 1 to max_plots do
                dispose(graph_buf[i])
        else
             dispose(graph_buf[1]);
  end;
  { ------------------------------------------- }
  { -- Concentration at point of Mix ---------- }
  { ------------------------------------------- }

  function mix(qUP,cUP,Qdn,Cdn: real): real;
  begin
       mix := (qUP*cUP + Qdn*Cdn) / (Qup + Qdn);
  end;

  { ------------------------------------------- }
  { -- Size in meters of increment length ----- }
  { ------------------------------------------- }

  function find_dx(reach: integer): real;
  var
     tlength,up,report,tot,temp : real;
     i              : longint;

  procedure set_x(var x : real);
  begin
       if x > 10 then x := 10
       else if x > 5 then x := 5
       else if x > 1 then x := 1
       else if x > 0.5 then x := 0.5
       else if x > 0.1 then x := 0.1
       else if x > 0.05 then x := 0.05
       else if x > 0.01 then x := 0.01
       else if x > 0.005 then x := 0.005
       else x := 0.001;
  end;

  begin
       tot := 0;
       for i := 1 to number_reaches do
           tot := tot + r_length[i];

       i := max_buffer_size;

       repeat
          report := tot / i;
          set_x(report);
          i := i - 1;
       until (tot / report) + 2*(number_reaches) < max_buffer_size;

       if ad[reach] <> 0 then
       begin
            next_dx := r_length[reach] / 500.0;
            set_x(next_dx);
            if next_dx > report then next_dx := report;
       end
       else
           next_dx := report;

       next_piece := round(report / next_dx);

       up := Upstream_river_mile;
       for i := 1 to reach - 1 do
           up := up - r_length[i];

       i := trunc(up / next_dx);
       tlength := up - (next_dx * i);

       if tlength <= 0 then tlength := next_dx;

       i := trunc(up / report);
       temp := report * i;

       if temp >= up then
          per_piece := next_piece
       else
           per_piece := round((up - temp - tlength) / next_dx) + 1;

       next_dx := next_dx * miles_to_meters;
       find_dx := tlength * miles_to_meters;
  end;

  { ------------------------------------------- }
  { -- Total Solids --------------------------- }
  { ------------------------------------------- }

  function find_m1 (m10,m2,ws,wrs,h1,u1,x : real) : real;
  var lnn : real;                           begin
       lnn := exp(( -ws * x) / ( h1 * u1) );
       find_m1 := m10 * lnn + ( ((wrs*m2)/ws) * (1-lnn));
  end;

  { ------------------------------------------- }
  { -- Disolved fraction ---------------------- }
  { ------------------------------------------- }

  function find_fd (m,pi : real) : real;
  begin
       find_fd := 1 /(1 + (m*pi));
  end;

  { ------------------------------------------- }
  { -- Sediment Capacity Factor --------------- }
  { ------------------------------------------- }

  function find_beta (m1,m2,h1,h2,fp1,fp2 : real) : real;
  begin
       find_beta := (m2 * h2 * fp1) / (m1 * h1 * fp2)
  end;

  { ------------------------------------------- }
  { -- Sedimentation burial velocity ---------- }
  { ------------------------------------------- }

  function find_wd (ws,m1,m2,wrs : real) : real;
  begin
       find_wd := (ws * m1 / m2) - wrs;
  end;

  { ------------------------------------------- }
  { -- R2 / R1 -------------------------------- }
  { ------------------------------------------- }

  function find_r2r1 (wd,wrs,fp2,fd2,kl,k2,partco1,partco2,h2 : real) : real;
  var r1,r2 : real;
  begin
            r2 := (wrs + wd) * fp2 + kl * (partco2/partco1) * fd2;    { 2 / 1 changed 10/8/90 - DC }
            r1 := (wrs + wd) * fp2 + kl * fd2 + k2 * h2;
            find_r2r1 := r2/r1;
  end;

  { ------------------------------------------- }
  { -- Total Toxicant in water ---------------- }
  { ------------------------------------------- }

  function find_ct1 (ws,wrs,ct10,kl,k1,x,u1,h1,h2,beta,fp1,fd1,fp2,fd2,r2r1,ad,q : real) : real;
  var kt : real;
  begin
       kt := k1 + (ws * fp1)/h1 + (kl * fd1)/h1 - ( ((beta*r2r1)/h2) * (wrs*fp2 + kl * fd2) );
       find_ct1 := ct10 * (1.0 - kt * x / u1) + (ad * x)/q;
{       find_ct1 := ct10 * exp(-1 * kt * x / u1); }
  end;

  { ------------------------------------------- }
  { -- Total Toxicant in sediment ------------- }
  { ------------------------------------------- }

  function find_ct2 (m1,m2,fp1,fp2,ct1,r2r1 : real) : real;
  begin
       find_ct2 := r2r1 * ct1 * m2 * fp1 / (m1 * fp2)
  end;

  { ------------------------------------------- }
  { -- Sets variables for SMPTOX, SMPTOX2 ----- }
  { ------------------------------------------- }
  var
     mod_m   : global_var;
     mod_h   : global_var;
     mod_kp  : global_var;
     mod_vl  : reach_var;
     mod_kd  : global_var;
     mod_pc  : global_var;
     mod_ws  : reach_var;
     mod_wrs : reach_var;
     mod_kl  : reach_var;
     mod_ad  : reach_var;

  procedure set_variables;
  var i,j : integer;
  begin
       for j := 0 to max_reaches do
       begin
            for i := 1 to 2 do
            begin
                 mod_m[i,j] := m[i,j];
                 mod_h[i,j] := h[i,j];
                 mod_kp[i,j] := kp[i,j];
                 mod_kd[i,j] := kd[i,j];
                 mod_pc[i,j] := pc[i,j];
            end;
            mod_vl[j] := vl[j];
            mod_ws[j] := ws[j];
            mod_wrs[j] := wrs[j];
            mod_kl[j] := kl[j];
            mod_ad[j] := ad[j];
       end;

       if model_type = SMPTOX2 then
            for j := 0 to max_reaches do
            begin
                 for i := 1 to 2 do
                    kp[i,j] := 0;

                 m[2,j] := 1;
                 h[1,j] := 3.28084;
                 vl[j] := 0;
                 kd[2,j] := 0;
                 wrs[j] := 0;
                 if pc[1,j] = 0 then pc[2,j] := 0
                 else pc[2,j] := 1;
                 h[2,j] := 1;
                 kl[j] := 0;
                 ad[j] := 0;
            end
       else
       if model_type = SMPTOX1 then
            for j := 0 to max_reaches do
            begin
                 for i := 1 to 2 do
                 begin
                      kp[i,j] := 0;
                      pc[i,j] := 0;
                 end;

                 m[1,j] := 0;
                 m[2,j] := 1;
                 h[1,j] := 1;
                 vl[j] := 0;
                 kd[2,j] := 0;
                 ws[j] := 1;
                 wrs[j] := 0;
                 h[2,j] := 1;
                 kl[j] := 0;
                 ad[j] := 0;
            end;
  end; { end set_variables}

  { ------------------------------------------- }
  { -- Unsets variables for SMPTOX, SMPTOX2 --- }
  { ------------------------------------------- }
  procedure unset_variables;
  var i,j : integer;
  begin
       for j := 0 to max_reaches do
       begin
            for i := 1 to 2 do
            begin
                 m[i,j] := mod_m[i,j];
                 h[i,j] := mod_h[i,j];
                 kp[i,j] := mod_kp[i,j];
                 kd[i,j] := mod_kd[i,j];
                 pc[i,j] := mod_pc[i,j];
            end;
            vl[j] := mod_vl[j];
            ws[j] := mod_ws[j];
            wrs[j] := mod_wrs[j];
            kl[j] := mod_kl[j];
            ad[j] := mod_ad[j];
       end;

  end; { end unset_variables }

  { ------------------------------------------- }
  { -- Find the Analytical solution ----------- }
  { ------------------------------------------- }

  function get_analytical(reach:integer;len :real;ft:boolean):real;
  var a , b : real;
  begin
         gettime(hour,min,sec,s100);
         a := hour * 3600 + min * 60 + sec + s100 / 100.0;
         get_analytical := find_analytical(
                              Kp[1,reach],
                              (Kd[1,reach]+vl[reach]),
                              kp[2,reach],
                              Kd[2,reach],
                              KL[reach],
                              U[reach] * feet_to_meters * secs_per_day,
                              initial_ct[reach],                                {water column total toxicant ug/L, at x=0}
                              Wrs[reach],                 {resuspension velocity m/day}
                              Ws[reach],                  {settling velocity m/day, must be >0}
                              initial_m1[reach],          {water column solids mg/L, at x=0}
                              m[2,reach],                 {sediment solids mg/L}
                              pc[1,reach],                {water column partition coefficient L/mg}
                              pc[2,reach],                {sediment partition coefficient L/mg}
                              H[1,reach]* feet_to_meters,  {water column depth m}
                              H[2,reach]*feet_to_meters,   {active sediment depth m}
                              ad[reach] * 1000.0 / (feet_to_meters * 5280),
                              total_Q[reach] * secs_per_day,
                              len,ft);

         gettime(hour,min,sec,s100);
         b := hour * 3600 + min * 60 + sec + s100 / 100.0;
         time_a := time_a + (b - a);


  end;

  { -------------------------------------------------------------------- }
  { -- Special Case solution to the general analytic, for pc[2,] = 0 --- }
  { -------------------------------------------------------------------- }

  procedure special_case(reach:integer;len: real);
  var aa : real;
  begin
       if (kl[reach] = 0) and (k[2,reach] = 0) then
       begin
            aa := k[1,reach] / (u[reach] * feet_to_meters * secs_per_day);
                        ct[2,reach] := 0;
            if aa <> 0 then
               run_ct := initial_ct[reach] * exp(-aa * len)
                         + (ad[reach] * 1000.0 / (feet_to_meters * 5280))
                         / (total_Q[reach] * secs_per_day * aa)
                         * ( 1 - exp(-aa * len))
            else
                run_ct := initial_ct[reach] + len * (ad[reach] * 1000.0
                / (feet_to_meters * 5280)) / (total_Q[reach] * secs_per_day);
       end
       else
       begin
            aa := (k[1,reach] * h[1,reach] * feet_to_meters  +  kl[reach]
                   -  kl[reach]*kl[reach]/(h[2,reach] * feet_to_meters * k[2,reach] + kl[reach]))
                  / (h[1,reach] * feet_to_meters * u[reach] * feet_to_meters * secs_per_day);
            if aa <> 0 then
               run_ct := initial_ct[reach] * exp(-aa * len)
                         + ((ad[reach] * 1000.0 / (feet_to_meters * 5280))
                         / (total_Q[reach] * secs_per_day * aa))
                         * ( 1 - exp(-aa * len))
            else
                run_ct := initial_ct[reach] + len * (ad[reach] * 1000.0 / (feet_to_meters * 5280))
                / (total_Q[reach] * secs_per_day);

            ct[2,reach] := (kl[reach] * run_ct / (h[2,reach]*feet_to_meters))
                        /  (k[2,reach] + kl[reach] / (h[2,reach] * feet_to_meters));
       end;
  end;

  { -------------------------------------------- }
  { -- Main program loop, analytical solution -- }
  { -------------------------------------------- }

  procedure solve_reach(reach:integer);

  var i,j                 : integer;
      dx,dxx,len,total_len    : real;      { in meters }
      run_m10             : real;      { suspended solids }
      beta,r2r1           : real;      { intermediate result }
      mile_mark           : real;
      out                 : text;
      burial              : real;
      first_time          : boolean;   { flag for end of reach }

  procedure fill_vars;
  begin
       if (model_type = SMPTOX3) then
       begin
            if pc[2,reach] > 0 then
               ct[2,reach] := find_ct2( run_m1,
                                m[2,reach],
                                fp[1,reach],
                                fp[2,reach],
                                run_ct,
                                r2r1 );
       end
       else ct[2,reach] := 0;

       cd[1,reach] := run_ct * fd[1,reach];
       cp[1,reach] := run_ct * fp[1,reach];
       cd[2,reach] := ct[2,reach] * fd[2,reach];
       cp[2,reach] := ct[2,reach] * fp[2,reach];

       if scaling_factor < 2000.0 then
          cp[2,reach] := cp[2,reach] *1000.0 / m[2,reach]  { for ug / mg }
       else if scaling_factor < 2000000.0 then
          cp[2,reach] := cp[2,reach]  / m[2,reach]  { for ug / mg }
       else if scaling_factor < 2000000000.0 then
          cp[2,reach] := cp[2,reach] / (1000.0 * m[2,reach])  { for ug / mg }
       else
          cp[2,reach] := (cp[2,reach] / m[2,reach]);  { for pg / mg }

       graph_buf[loop_here]^[0,last_buf] := mile_mark - (len / miles_to_meters);
       graph_buf[loop_here]^[1,last_buf] := run_ct;
       graph_buf[loop_here]^[2,last_buf] := ct[2,reach];
       graph_buf[loop_here]^[3,last_buf] := cd[1,reach];
       graph_buf[loop_here]^[4,last_buf] := cd[2,reach];
       graph_buf[loop_here]^[5,last_buf] := cp[1,reach];
       graph_buf[loop_here]^[6,last_buf] := cp[2,reach];
       graph_buf[loop_here]^[7,last_buf] := run_m1;

       if model_type = SMPTOX3 then
       begin
            burial := exp(mday_to_mmyear * ((run_m1/m[2,reach])*ws[reach] - wrs[reach]));
            if burial > 4.5 then burial := 4.5;
            if burial < -4.5 then burial := -4.5;
            graph_buf[loop_here]^[8,last_buf] := burial;
       end;
end;

  begin

       Mile_mark := upstream_river_mile;
       for i := 1 to reach-1 do
           mile_mark := mile_mark - r_length[i];

       total_len := r_length[reach] * miles_to_meters;

       dx := find_dx(reach);

       run_m1 := mix( Total_Q[reach-1],
                      run_m1,
                      q[reach]*mgd_to_ls,
                      m[1,reach] );

       initial_m1[reach] :=  run_m1;

       run_ct := mix( Total_Q[reach-1],
                      run_ct,
                      q[reach]*mgd_to_ls,
                      dt[1,reach] );

       initial_ct[reach] := run_ct;

       this_piece := 0;
       first_time := TRUE;

       len := 0;

       while len <= total_len do
       begin
            if len = 0 then
               dxx := 0
            else dxx := dx;

            if (model_type <> SMPTOX3) and ((ws[reach] = 0) or (run_m1=0)) then
            begin
                 fd[1,reach] := find_fd( run_m1,pc[1,reach]);
                 fp[1,reach] := 1 - fd[1,reach];
                 run_ct := run_ct * exp(-kd[1,reach]*fd[1,reach]*dxx
                               / (u[reach]*feet_to_meters*secs_per_day));

            end
            else
            begin
                 run_m1 := find_m1( initial_m1[reach],
                                    m[2,reach],                           { Bed solids are constant over all reaches }
                                    ws[reach],                            { settling velocity (m/day) }
                                    wrs[reach],                           { resuspension velocity (m/day) }
                                    h[1,reach]*feet_to_meters,            { depth (meters) }
                                    u[reach]*feet_to_meters*secs_per_day, { velocity (meters per day) }
                                    len);                                 { distance since last, not len }

                 fd[1,reach] := find_fd( run_m1,pc[1,reach]);
                 fd[2,reach] := find_fd( m[2,reach],pc[2,reach]);

                 fp[1,reach] := 1-fd[1,reach];
                 k[1,reach] :=  (kd[1,reach] + vl[reach]) * fd[1,reach] + kp[1,reach] * fp[1,reach];
                 fp[2,reach] := 1-fd[2,reach];
                 k[2,reach] :=  kd[2,reach] * fd[2,reach] + kp[2,reach] * fp[2,reach];

                 if pc[2,reach] > 0 then
                 begin
                      wd[reach] := find_wd( ws[reach],
                                  run_m1,
                                  m[2,reach],
                                  wrs[reach] );

                      r2r1 := find_r2r1( wd[reach],
                               wrs[reach],
                               fp[2,reach],
                               fd[2,reach],
                               kl[reach],
                               k[2,reach],
                               pc[1,reach],
                               pc[2,reach],
                               h[2,reach] * feet_to_meters );

                      if len = 0 then
                      begin
                           run_ct := get_analytical(reach,total_len,True);
                           run_ct := initial_ct[reach];
                      end
                      else
                          run_ct := get_analytical(reach,len,False);
                 end
                 else
                     special_case(reach,len);
            end; { not ws = 0 }


            fill_vars;

            if len = 0 then
            begin
                 len := len + dx;
                 this_piece := 2;
            end
            else
                this_piece := 1;

            dx := next_dx;

            while this_piece <= per_piece do
            begin
                 len := len + dx;
                 this_piece := this_piece + 1;
            end;

            per_piece := next_piece;

            if first_time and ((abs(len - total_len) < 1) or (len > total_len)) then
            begin
                 len := total_len;
                 first_time := FALSE;
            end;

            total_buf := total_buf +1;
            last_buf := last_buf + 1;

       end; { end while }

  end; { end solve_reach }

  { -------------------------------------------- }
  { -- Main program loop, numerical solution --- }
  { -------------------------------------------- }

  procedure loop_reach(reach:integer);

  var i,j                 : integer;
      dx,dxx,len,total_len    : real;      { in meters }
      aa                  : real;      { temporary }
      run_m10             : real;      { suspended solids }
      beta,r2r1           : real;      { intermediate result }
      mile_mark           : real;
      out                 : text;
      first_time          : boolean;   { flag for end of reach }
  begin

       Mile_mark := upstream_river_mile;
       for i := 1 to reach-1 do
           mile_mark := mile_mark - r_length[i];

       total_len := r_length[reach] * miles_to_meters;

       dx := find_dx(reach);

       run_m1 := mix( Total_Q[reach-1],
                      run_m1,
                      q[reach]*mgd_to_ls,
                      m[1,reach] );

       initial_m1[reach] :=  run_m1;

       run_ct := mix( Total_Q[reach-1],
                      run_ct,
                      q[reach]*mgd_to_ls,
                      dt[1,reach] );

       initial_ct[reach] := run_ct;

       this_piece := 0;
       first_time := TRUE;

       len := 0;

       while len <= total_len do
       begin
            if len = 0 then
               dxx := 0
            else dxx := dx;

            run_m1 := find_m1( initial_m1[reach],
                               m[2,reach],                           { Bed solids are constant over all reaches }
                               ws[reach],                            { settling velocity (m/day) }
                               wrs[reach],                           { resuspension velocity (m/day) }
                               h[1,reach]*feet_to_meters,            { depth (meters) }
                               u[reach]*feet_to_meters*secs_per_day, { velocity (meters per day) }
                               len);                                 { distance since last, not len }


            fd[1,reach] := find_fd( run_m1,pc[1,reach]);
            fd[2,reach] := find_fd( m[2,reach],pc[2,reach]);

            fp[1,reach] := 1-fd[1,reach];
            k[1,reach] :=  (kd[1,reach] + vl[reach]) * fd[1,reach] + kp[1,reach] * fp[1,reach];
            fp[2,reach] := 1-fd[2,reach];
            k[2,reach] :=  kd[2,reach] * fd[2,reach] + kp[2,reach] * fp[2,reach];

            if pc[2,reach] > 0 then
            begin
                 beta := find_beta( run_m1,
                               m[2,reach],
                               h[1,reach] * feet_to_meters,
                               h[2,reach] * feet_to_meters,
                               fp[1,reach],
                               fp[2,reach] );

                 wd[reach] := find_wd( ws[reach],
                                  run_m1,
                                  m[2,reach],
                                  wrs[reach] );

                 r2r1 := find_r2r1( wd[reach],
                               wrs[reach],
                               fp[2,reach],
                               fd[2,reach],
                               kl[reach],
                               k[2,reach],
                               pc[1,reach],
                               pc[2,reach],
                               h[2,reach] * feet_to_meters );

                 run_ct := find_ct1( ws[reach],
                                wrs[reach],
                                run_ct,
                                kl[reach],
                                k[1,reach],
                                dxx,  { dx distance since last, or total length from discharge ? }
                                u[reach] * feet_to_meters * secs_per_day,
                                h[1,reach] * feet_to_meters,
                                h[2,reach] * feet_to_meters,
                                beta,
                                fp[1,reach],
                                fd[1,reach],
                                fp[2,reach],
                                fd[2,reach],
                                r2r1,
                                ad[reach] * 1000.0 / (feet_to_meters * 5280),
                                total_Q[reach] * secs_per_day);

                 ct[2,reach] := find_ct2( run_m1,
                                     m[2,reach],
                                     fp[1,reach],
                                     fp[2,reach],
                                     run_ct,
                                     r2r1 );
            end
            else
                special_case(reach,len);
                { -- Use the special case solution worked out by JH -- }

            cd[1,reach] := run_ct * fd[1,reach];
            cp[1,reach] := run_ct * fp[1,reach];
            cd[2,reach] := ct[2,reach] * fd[2,reach];
            cp[2,reach] := ct[2,reach] * fp[2,reach];
            if scaling_factor < 2000.0 then
               cp[2,reach] := cp[2,reach] *1000.0 / m[2,reach]  { for ug / mg }
            else if scaling_factor < 2000000.0 then
               cp[2,reach] := cp[2,reach]  / m[2,reach]  { for ug / mg }
            else if scaling_factor < 2000000000.0 then
               cp[2,reach] := cp[2,reach] / (1000.0 * m[2,reach])  { for ug / mg }
            else
               cp[2,reach] := (cp[2,reach] / m[2,reach]);  { for pg / mg }


            graph_buf[loop_here]^[0,last_buf] := mile_mark - (len / miles_to_meters);
            graph_buf[loop_here]^[1,last_buf] := run_ct;
            graph_buf[loop_here]^[2,last_buf] := ct[2,reach];
            graph_buf[loop_here]^[3,last_buf] := cd[1,reach];
            graph_buf[loop_here]^[4,last_buf] := cd[2,reach];
            graph_buf[loop_here]^[5,last_buf] := cp[1,reach];
            graph_buf[loop_here]^[6,last_buf] := cp[2,reach];
            graph_buf[loop_here]^[7,last_buf] := run_m1;
            graph_buf[loop_here]^[8,last_buf] := exp((run_m1/m[2,reach])*ws[reach] - wrs[reach]);

            this_piece := this_piece + 1;

            if len = 0 then
            begin
(*              this_piece := get_analytical(reach,total_len,True); *)
                 this_piece := 0;
(*               graph_buf[loop_here]^[8,last_buf] := initial_ct[reach]; *)
                 total_buf := total_buf +1;
                 last_buf := last_buf + 1;
            end
            else
            if this_piece >= per_piece then
            begin
                 per_piece := next_piece;
                 this_piece := 0;
(*               graph_buf[loop_here]^[8,last_buf] := get_analytical(reach,len,False); *)
                 last_buf := last_buf + 1;
                 total_buf := total_buf + 1;
            end;

            { baby step at bottom of reach }

            if (len <> 0) and first_time then dx := next_dx;

            if (len + dx > total_len) and first_time and (len + 1 < total_len) then
            begin
                 first_time := FALSE;
                 dx  := total_len - len;     { correction }
                 len := total_len;
            end
            else
                len := len + dx;

       end; { end while }
       if this_piece <> 0 then
       begin
(*          graph_buf[loop_here]^[8,last_buf] := get_analytical(reach,total_len,false); *)
            last_buf := last_buf + 1;
            total_buf := total_buf + 1;
       end;

  end; { end loop_reach }

  { ------------------------------------------- }
  { -- Main Model Driver ---------------------- }
  { ------------------------------------------- }
  procedure run_model;
  var out        : text;
      i,j,iter   : integer;
      ii,jj      : longint;
      ttemp,stemp: string[5];
      fctr       : real;
      total_tic  : real;
      tic        : real;
      nout       : integer;
      flag       : boolean;
      sens_u     : reach_var;
      sens_h     : global_var;
      sens_kd    : global_var;
      sens_vl    : reach_var;
      sens_kp    : global_var;
      sens_ws    : reach_var;
      sens_wrs   : reach_var;
      sens_kl    : reach_var;
      sens_pc    : global_var;
      sens_m     : global_var;
      sens_ct    : global_var;
      ftim,ntim  : real;
  begin
       set_variables;
       for i := 0 to max_graphs do
         sense_name[i] := '';
       sense_name[0] := 'Baseline Simulation';

       time_a := 0;
       time_n := 0;

       iter := 1;
       while iter <> 4 do
       begin
            getmenuchoice(iter, 'SMPTOX3 RUN MENU/'+
                                'Calibration \ Verification/'+
                                'Waste Load Allocation/'+
                                'Sensitivity Analysis/'+
                                'Exit to Main Menu');

            if( iter = ESCAPE) or (iter = 4) then
            begin
                 unset_variables;
                 exit;
            end;

            high_model := model_type;

            if iter = WLA_PLOT then
            begin
                 i:=modify_discharge;
                 if i = ESCAPE then
                 begin
                      { unmodify_discharge; }
                      unset_variables;
                      exit;
                 end;
            end;

            { store off variables }
            if iter = SENSITIVITY_PLOT then
            begin
                 for i := 0 to max_reaches do
                 begin
                      sens_u[i] := u[i];
                      sens_ws[i] := ws[i];
                      sens_wrs[i] := wrs[i];
                      sens_kl[i] := kl[i];
                      sens_vl[i] := vl[i];

                      for j := 1 to 2 do
                      begin
                           sens_h[j,i] := h[j,i];
                           sens_kd[j,i] := kd[j,i];
                           sens_kp[j,i] := kp[j,i];
                           sens_pc[j,i] := pc[j,i];
                           sens_m[j,i] := m[j,i];
                           sens_ct[j,i] := ct[j,i];
                      end;
                 end;

                 if edit_sense = escape then
                 begin
                      unset_variables;
                      exit;
                 end;

                 if (max_plots <> 0) or (nplots <> 0) then
                 begin
                      clean_up;
                 end;

                 nplots := 1;
                 max_plots := 1;
                 i := 1;
                 while i <= 14 do
                 if sense_out[i] >= 100.0 then
                 begin
                      Error_message('Percent Change May Not Exceed 100');
                      if edit_sense = escape then
                      begin
                           nplots := 0;
                           max_plots := 0;
                           unset_variables;
                           exit;
                      end;
                      i := 1;
                 end
                 else i := i +1;

                 for i := 1 to 14 do
                     if sense_out[i] <> 0 then max_plots := max_plots + 2;
                 if max_plots > max_graphs then max_plots := max_graphs;

                 jj := sizeof(graphics_buffer);
                 for i := 1 to (max_plots) do
                     if MaxAvail <= jj then
                     begin
                          Error_message('Not Enough Memory to Simulate');
                          nplots := i-1;
                          max_plots := i-1;
                          unset_variables;
                          graph_type := iter;                 { store for later }
                          exit;
                     end
                     else
                         new(graph_buf[i]);
                 message( 'Baseline Analysis','',TRUE);
            end
            else
            begin
                 if (max_plots <> 0) or (nplots <> 0) then
                 begin
                      clean_up;
                 end;
                 nplots := 1;
                 max_plots := 1;
                 ii := MaxAvail;
                 jj := sizeof(graphics_buffer);
                 if ii <= jj then
                 begin
                      Error_message('Not Enough Memory to Simulate');
                      unset_variables;
                      nplots := 0;
                      max_plots := 0;
                      graph_type := iter;                 { store for later }
                      exit;
                 end;
                 new(graph_buf[1]);
                 message( 'Simulation In Progress','',TRUE)
            end;

            total_q[0] := q[0]*cfs_to_ls;
            for i := 1 to number_reaches do
                total_q[i] := total_q[i-1] + q[i]*mgd_to_ls;

            loop_here := 1;
            nout := 0;
            while( loop_here <= max_plots) do
            begin

                 total_buf := 0;
                 last_buf := 0;

                 run_m1 := m[1,0];    { reset suspended solids to upstream value }
                 run_ct := ct[1,0];   { reset total toxicant to upstream value }
                 str(number_reaches,stemp);

                   { time it  }
                 gettime(hour,min,sec,s100);
                 ftim := hour * 3600 + min * 60 + sec + s100 / 100.0;

                 for i := 1 to number_reaches do
                 begin
                      str(i,ttemp);
                      message( '','Simulating in reach '+ttemp+' of '+stemp,False);
                      if ad[i] <> 0 then
                         loop_reach(i)     { run the numeric solution }
                      else solve_reach(i); { run the analytical solution }
                 end;
                     { more timing }
                 gettime(hour,min,sec,s100);
                 ntim := hour * 3600 + min * 60 + sec + s100 / 100.0;
                 time_n := ntim - ftim - time_a;

                 case iter of
                      Observation_Plot :
                          begin
                               show_graph(Observation_Plot);
                               loop_here := Max_graphs + 1;
                          end;

                      WLA_PLot :
                          begin
                               show_graph(WLA_PLOT);
                               loop_here := Max_graphs + 1;
                          end;

                      Sensitivity_Plot :
                          begin
                               if loop_here mod 2 = 1 then
                               begin
                                    fctr := 1.0;
                                    nout := nout + 1;
                               end
                               else
                                    fctr := -1.0;

                               flag := true;
                               while (flag = true) and (nout <= 14) and (fctr = 1) do
                               begin
                                    if sense_out[nout] > 0.0001 then flag := FALSE   { be careful of round-off }
                                    else nout := nout + 1;
                               end;

                               if nout = 15 then loop_here := max_graphs +1
                               else
                               begin
                                    for i := 0 to max_reaches do
                                    begin
                                         u[i] := sens_u[i];
                                         ws[i] := sens_ws[i];
                                         wrs[i] := sens_wrs[i];
                                         kl[i] := sens_kl[i];
                                         vl[i] := sens_vl[i];

                                         for j := 1 to 2 do
                                         begin
                                              h[j,i] := sens_h[j,i];
                                              kp[j,i] := sens_kp[j,i];
                                              kd[j,i] := sens_kd[j,i];
                                              pc[j,i] := sens_pc[j,i];
                                              m[j,i] := sens_m[j,i];
                                              ct[j,i] := sens_ct[j,i];
                                         end;
                                    end;
                                    for i := 1 to number_reaches do
                                    case nout of
                                       1 : begin
                                           u[i] := sens_u[i] + (fctr*sens_u[i]*sense_out[1]*0.01);
                                           if fctr = 1 then
                                              sense_name[loop_here] := 'Velocity, Upper Bound Analysis'
                                           else
                                              sense_name[loop_here] := 'Velocity,  Lower Bound Analysis';
                                       end;
                                       2 : begin
                                           h[1,i] := sens_h[1,i] + (fctr*sens_h[1,i]*sense_out[2]*0.01);
                                           if fctr = 1 then
                                              sense_name[loop_here] := 'River Depth, Upper Bound Analysis'
                                           else
                                               sense_name[loop_here] := 'River Depth,  Lower Bound Analysis';
                                       end;
                                       3 : begin
                                           h[2,i] := sens_h[2,i] + (fctr*sens_h[2,i]*sense_out[3]*0.01);
                                           if fctr = 1 then
                                              sense_name[loop_here] := 'Active Bed Depth, Upper Bound Analysis'
                                           else
                                              sense_name[loop_here] := 'Active Bed Depth, Lower Bound Analysis';
                                       end;
                                       4 : begin
                                           kp[1,i] := sens_kp[1,i] + (fctr*sens_kp[1,i]*sense_out[4]*0.01);
                                           kd[1,i] := sens_kd[1,i] + (fctr*sens_kd[1,i]*sense_out[4]*0.01);
                                           if fctr = 1 then
                                              sense_name[loop_here] := 'Water Decay Rates,  Upper Bound Analysis'
                                           else
                                              sense_name[loop_here] := 'Water Decay Rates,  Lower Bound Analysis';
                                       end;
                                       5 : begin
                                           vl[i] := sens_vl[i] + (fctr*sens_vl[i]*sense_out[5]*0.01);
                                           if fctr = 1 then
                                              sense_name[loop_here] := 'Volatilization,  Upper Bound Analysis'
                                           else
                                              sense_name[loop_here] := 'Volatilization,  Lower Bound Analysis';
                                       end;
                                       6 : begin
                                           kp[2,i] := sens_kp[2,i] + (fctr*sens_kp[2,i]*sense_out[6]*0.01);
                                           kd[2,i] := sens_kd[2,i] + (fctr*sens_kd[2,i]*sense_out[6]*0.01);
                                           if fctr = 1 then
                                              sense_name[loop_here] := 'Bed Decay Rates, Upper Bound Analysis'
                                           else
                                              sense_name[loop_here] := 'Bed Decay Rates, Lower Bound Analysis';
                                       end;
                                       7 : begin
                                           ws[i] := sens_ws[i] + (fctr*sens_ws[i]*sense_out[7]*0.01);
                                           if fctr = 1 then
                                              sense_name[loop_here] := 'Settling Velocity, Upper Bound Analysis'
                                           else
                                              sense_name[loop_here] := 'Settling Velocity, Lower Bound Analysis';
                                       end;
                                       8 : begin
                                           wrs[i] := sens_wrs[i] + (fctr*sens_wrs[i]*sense_out[8]*0.01);
                                           if fctr = 1 then
                                              sense_name[loop_here] := 'Resuspension Velocity, Upper Bound Analysis'
                                           else
                                              sense_name[loop_here] := 'Resuspension Velocity, Lower Bound Analysis';
                                       end;
                                       9 : begin
                                           pc[1,i] := sens_pc[1,i] + (fctr*sens_pc[1,i]*sense_out[9]*0.01);
                                           if fctr = 1 then
                                              sense_name[loop_here] := 'Water Partition, Upper Bound Analysis'
                                           else
                                              sense_name[loop_here] := 'Water Partition, Lower Bound Analysis';
                                       end;
                                       10 : begin
                                           pc[2,i] := sens_pc[2,i] + (fctr*sens_pc[2,i]*sense_out[10]*0.01);
                                           if fctr = 1 then
                                              sense_name[loop_here] := 'Bed Partition, Upper Bound Analysis'
                                           else
                                              sense_name[loop_here] := 'Bed Partition,  Lower Bound Analysis';
                                       end;
                                       11 : begin
                                            kl[i] := sens_kl[i] + (fctr*sens_kl[i]*sense_out[11]*0.01);
                                            if fctr = 1 then
                                               sense_name[loop_here] := 'Diffusive Exchange, Upper Bound Analysis'
                                            else
                                               sense_name[loop_here] := 'Diffusive Exchange, Lower Bound Analysis';
                                       end;
                                       12 : begin
                                            m[2,i] := sens_m[2,i] + (fctr*sens_m[2,i]*sense_out[12]*0.01);
                                            if fctr = 1 then
                                               sense_name[loop_here] := 'Bed Solids, Upper Bound Analysis'
                                            else
                                                sense_name[loop_here] := 'Bed Solids, Lower Bound Analysis';
                                       end;
                                       13 : begin
                                            ct[1,0] := sens_ct[1,0] + (fctr*sens_ct[1,0]*sense_out[13]*0.01);
                                            if fctr = 1 then
                                               sense_name[loop_here] := 'Upstream Toxicant, Upper Bound Analysis'
                                            else
                                                sense_name[loop_here] := 'Upstream Toxicant, Lower Bound Analysis';
                                       end;
                                       14 : begin
                                            m[1,0] := sens_m[1,0] + (fctr*sens_m[1,0]*sense_out[14]*0.01);
                                            if fctr = 1 then
                                               sense_name[loop_here] := 'Upstream Suspended Solids, Upper Bound Analysis'
                                            else
                                                sense_name[loop_here] := 'Upstream Suspended Solids.  Lower Bound Analysis';
                                       end;
                                    end; { end case }

                                    message(sense_name[loop_here],'',False);
                                    loop_here := loop_here + 1;
                               end;
                          end;  { case = 3 }
                      end; { end case }
                      graph_type := iter;                 { store for later }
            end; { end while loop_here }

            if iter = Sensitivity_Plot then
            begin
                 for i := 0 to max_reaches do
                 begin
                      u[i] := sens_u[i];
                      ws[i] := sens_ws[i];
                      wrs[i] := sens_wrs[i];
                      kl[i] := sens_kl[i];
                      vl[i] := sens_vl[i];

                      for j := 1 to 2 do
                      begin
                           h[j,i] := sens_h[j,i];
                           kp[j,i] := sens_kp[j,i];
                           kd[j,i] := sens_kd[j,i];
                           pc[j,i] := sens_pc[j,i];
                           m[j,i] := sens_m[j,i];
                           ct[j,i] := sens_ct[j,i];
                      end;
                 end;
                 message('Graphing Results','Loading Graphics Information From Disk',False);
                 show_graph(Sensitivity_Plot);
            end; { end if sensitivity_plot }


          {  if graph_type = WLA_Plot then
               unmodify_discharge; }

       end; { end while menu }

       unset_variables;

  end;  { end run_model }


  { ------------------------------------------- }
  { -- Model Driver for Multiple File case ---- }
  { ------------------------------------------- }
  procedure many_model(file_chosen,been_before:boolean);
  var out              : text;
      i,j              : integer;
      buf,stemp,ttemp  : string[3];
      oldreaches  : reach_var;

      procedure show_mscreen;
      var i,j,k,nff : integer;
          title     : textstr;
      begin
            ClearScreen;
            cursor(off);
            str(max_graphs,buf);
            title := 'FILES CHOSEN:  ('+buf+' max)';
            j := length(title);
            i := trunc(j/2)+1;
            linetype := 2;

            if nf > 1 then nff := nf -1
               else nff := 1;

            textbox(8,40-i,11+nff,40+i+2);
            textbox(10,40-i,11+nff,40+i+2);
            showstring(title,40-i+2,9,j,OFF);

            for k := 1 to nff do
                showstring(files[k],40-i+2,10+k,j,off);

            linetype := 1;
            textlineH(25,1,79);
            showstring('   - To Add File      - To Delete File      - To Run Simulation',10,25,30,off);
            showstring('F1',10,25,2,ON);
            showstring('F2',29,25,2,ON);
            showstring('F10',50,25,3,ON);
      end;  { end show_mscreen }

  begin { many_model }

       if not been_before then
       begin
            for i := 1 to Max_graphs do
            begin
                 files[i] := '*.st3';
                 directories[i] := in_directory;
            end;

            if file_chosen then
            begin
                 files[1] := infile_name;
                 directories[1] := in_directory;
                 nf := 2;
            end
            else
            begin
                 files[1] := 'None Selected';
                 nf := 1;
            end;
       end;

       code := 1;

       while (code <> -68) and (code <> 27) do
       begin
            code := 1;
            show_mscreen;
            while (code <> -68) and (code <> 27) and
                  (code <> -59) and (code <> -60) do
                      getchar;

            if code = -60 then
               if nf <> 1 then
               begin
                    nf := nf - 1;
                    if nf = 1 then files[nf] := 'None Selected'
                    else files[nf] := '*.st3';
               end;

            if (code = -59) and (nf <= Max_graphs) then
            begin
                 files[nf] := '*.st3';
                 if get_file_name(directories[nf],files[nf],In_Put) <> '' then
                    nf := nf + 1
                 else files[nf] := '*.st3';
                 code := 1;
            end;

       end;

       if (code = 27) or (nf = 1) then exit;

       if (nplots <> 0) or (max_plots <> 0) then
       begin
            clean_up;
            nplots := 0;
            max_plots := 0;
       end;
       graph_type := many_plot;
       nplots := nf -1;
       max_plots := nf - 1;

      for  i := 1 to nplots do
      if MaxAvail <= sizeof(graphics_buffer) then
      begin
           Error_message('Not Enough Memory to Simulate');
           nplots := i-1;
           max_plots := i-1;

           { ----- Code added 1/23/92 to free unused graph buffers ----- }
           { ----- when aborting due to insufficient memory        ----- }

           for j := 1 to i-1 do
               Dispose(Graph_buf[J]);

           exit;
      end
      else
         new(graph_buf[i]);

       for loop_here := 1 to nplots do
       begin
            load_data(files[loop_here]);
            infile_name := files[loop_here];
            if loop_here = 1 then
               high_model := model_type
            else if model_type > high_model then
               high_model := model_type;
            set_variables;
            if loop_here <> 1 then
                 for i := 1 to  number_reaches do
                     if r_length[i] <> oldreaches[i] then
                     begin
                          Error_message('Selected Files Are Not Comparable.  Simulation Aborted');
                          unset_variables;
                          exit;
                     end;

            for i := 1 to number_reaches do
            oldreaches[i] := r_length[i];

            total_buf := 0;
            last_buf := 0;

            message('Analyzing file '+files[loop_here],'',TRUE);
            total_q[0] := q[0]*cfs_to_ls;
            for i := 1 to number_reaches do
                total_q[i] := total_q[i-1] + q[i]*mgd_to_ls;

             run_m1 := m[1,0];    { reset suspended solids to upstream value }
             run_ct := ct[1,0];   { reset total toxicant to upstream value }
             str(number_reaches,stemp);

             for i := 1 to number_reaches do
             begin
                  str(i,ttemp);
                  message( '','Simulating in reach '+ttemp+' of '+stemp,False);
                  if ad[i] <> 0 then
                     loop_reach(i)     { run the numeric solution }
                  else solve_reach(i); { run the analytical solution }
              end;

       end; { end for loop_here }
       message('Graphing Results','Loading Graphics Information From Disk',False);
       graph_type := many_plot;
       show_graph(Many_Plot);

  end;

end.   { end implementation }
